<?php

declare(strict_types=1);

/*
 * This file is part of the RegexParser package.
 *
 * (c) Younes ENNAJI <younes.ennaji.pro@gmail.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace RegexParser\Lint\Command;

use RegexParser\Cli\Command\AbstractCommand;
use RegexParser\Cli\Command\CommandInterface;
use RegexParser\Cli\Command\HelpCommand;
use RegexParser\Cli\Input;
use RegexParser\Cli\Output;
use RegexParser\Lint\Formatter\ConsoleFormatter;
use RegexParser\Lint\Formatter\FormatterRegistry;
use RegexParser\Lint\Formatter\LinkFormatter;
use RegexParser\Lint\Formatter\OutputConfiguration;
use RegexParser\Lint\Formatter\RelativePathHelper;
use RegexParser\Lint\PhpRegexPatternSource;
use RegexParser\Lint\RegexAnalysisService;
use RegexParser\Lint\RegexLintReport;
use RegexParser\Lint\RegexLintRequest;
use RegexParser\Lint\RegexLintService;
use RegexParser\Lint\RegexPatternSourceCollection;
use RegexParser\ReDoS\ReDoSConfirmOptions;
use RegexParser\ReDoS\ReDoSSeverity;

final class LintCommand extends AbstractCommand implements CommandInterface
{
    public function __construct(
        private readonly HelpCommand $helpCommand,
        private readonly LintConfigLoader $configLoader,
        private readonly LintDefaultsBuilder $defaultsBuilder,
        private readonly LintArgumentParser $argumentParser,
        private readonly LintExtractorFactory $extractorFactory,
        private readonly LintOutputRenderer $outputRenderer,
    ) {}

    public function getName(): string
    {
        return 'lint';
    }

    public function getAliases(): array
    {
        return [];
    }

    public function getDescription(): string
    {
        return 'Lint regex patterns in PHP source code';
    }

    public function run(Input $input, Output $output): int
    {
        $lintConfigResult = $this->configLoader->load();
        if (null !== $lintConfigResult->error) {
            $output->write($output->error('Error: '.$lintConfigResult->error."\n"));

            return 1;
        }

        $lintDefaults = $this->defaultsBuilder->build($lintConfigResult->config ?? []);
        $lintConfigFiles = $lintConfigResult->files ?? [];

        $parsed = $this->argumentParser->parse($input->args, $lintDefaults);
        if ($parsed->help) {
            return $this->helpCommand->run(new Input('help', [], $input->globalOptions, []), $output);
        }
        if (null !== $parsed->error) {
            $output->write($output->error('Error: '.$parsed->error."\n"));
            $output->write("Usage: regex lint [paths...] [--exclude <path>] [--min-savings <n>] [--jobs <n>] [--format console|json|github|checkstyle|junit] [--output <file>] [--baseline <file>] [--generate-baseline] [--no-redos] [--redos-mode=off|theoretical|confirmed] [--redos-threshold=low|medium|high|critical] [--redos-no-jit] [--no-validate] [--no-optimize] [--verbose|--debug|--quiet]\n");

            return 1;
        }

        $arguments = $parsed->arguments;
        if (null === $arguments) {
            $output->write($output->error("Error: Invalid lint arguments\n"));

            return 1;
        }

        $paths = $arguments->paths;
        $exclude = $arguments->exclude;
        $minSavings = (int) $arguments->minSavings;
        $verbosity = $arguments->verbosity;
        $format = $arguments->format;
        $quiet = $arguments->quiet;
        $checkRedos = $arguments->checkRedos;
        if ('off' === $arguments->redosMode) {
            $checkRedos = false;
        }
        $checkValidation = $arguments->checkValidation;
        $checkOptimizations = $arguments->checkOptimizations;
        $jobs = (int) $arguments->jobs;
        if (-1 === $jobs) {
            // Auto-detect optimal number of jobs
            $jobs = self::detectCpuCount();
        } elseif ($jobs < 1) {
            $jobs = 1; // Minimum 1 job
        }
        $outputFile = $arguments->output;

        if ([] === $paths) {
            $paths = ['.'];
        }
        if ([] === $exclude && !\array_key_exists('exclude', $lintDefaults)) {
            $exclude = ['vendor'];
        }

        $regex = $this->createRegex($output, $input->regexOptions);
        if (null === $regex) {
            return 1;
        }

        $config = match ($verbosity) {
            OutputConfiguration::VERBOSITY_QUIET => OutputConfiguration::quiet(),
            OutputConfiguration::VERBOSITY_VERBOSE => OutputConfiguration::verbose(),
            OutputConfiguration::VERBOSITY_DEBUG => OutputConfiguration::debug(),
            default => new OutputConfiguration(ansi: $output->isAnsi()),
        };

        $confirmOptions = $arguments->redosNoJit ? new ReDoSConfirmOptions(disableJit: true) : null;
        $analysis = new RegexAnalysisService(
            $regex,
            redosThreshold: $arguments->redosThreshold ?? ReDoSSeverity::HIGH->value,
            redosMode: $arguments->redosMode,
            redosConfirmOptions: $confirmOptions,
        );

        $formatterRegistry = new FormatterRegistry();
        if (!$formatterRegistry->has($format)) {
            $output->write($output->error(\sprintf('Unknown format: %s. Available formats: %s', $format, implode(', ', $formatterRegistry->getNames()))."\n"));

            return 1;
        }
        $formatter = $formatterRegistry->get($format);

        if ('console' === $format) {
            $linkFormatter = '' !== $arguments->ide
                ? new LinkFormatter($arguments->ide, new RelativePathHelper())
                : null;
            $formatter = new ConsoleFormatter($analysis, $config, $arguments->ide, $linkFormatter);
        }

        $extractor = $this->extractorFactory->create();
        $sources = new RegexPatternSourceCollection([
            new PhpRegexPatternSource($extractor),
        ]);
        $lint = new RegexLintService($analysis, $sources);

        if ('console' === $format && OutputConfiguration::VERBOSITY_QUIET !== $verbosity) {
            $output->write($this->outputRenderer->renderBanner($output, $jobs, $lintConfigFiles));
        }

        $collectionProgress = null;
        $startTime = (float) microtime(true);
        $collectionStartTime = $startTime;
        $fileCount = 0;

        if ('console' === $format && $config->shouldShowProgress()) {
            $output->write('  '.$output->dim('[1/2] Scanning files')."\n");
            $collectionStarted = false;
            $lastCount = 0;
            $collectionProgress = static function (int $current, int $total) use (&$collectionStarted, &$lastCount, $output, &$fileCount): void {
                if ($total <= 0) {
                    return;
                }

                $fileCount = $total;

                if (!$collectionStarted) {
                    $output->progressStart($total);
                    $collectionStarted = true;
                }

                $advance = $current - $lastCount;
                if ($advance > 0) {
                    $output->progressAdvance($advance);
                    $lastCount = $current;
                }

                if ($current >= $total) {
                    $output->progressFinish();
                }
            };
        }

        try {
            $request = new RegexLintRequest(
                paths: $paths,
                excludePaths: $exclude,
                minSavings: $minSavings,
                checkValidation: $checkValidation,
                checkRedos: $checkRedos,
                checkOptimizations: $checkOptimizations,
                analysisWorkers: $jobs,
                optimizations: $arguments->optimizations,
            );
            $patterns = $lint->collectPatterns($request, $collectionProgress);
        } catch (\Throwable $e) {
            $output->write($output->error("Failed to collect patterns: {$e->getMessage()}\n"));

            return 1;
        }

        $collectionTime = (float) microtime(true) - $collectionStartTime;

        if ([] === $patterns) {
            if ('console' === $format) {
                $this->outputRenderer->renderSummary($output, ['errors' => 0, 'warnings' => 0, 'optimizations' => 0], true);
            } else {
                $emptyReport = new RegexLintReport([], ['errors' => 0, 'warnings' => 0, 'optimizations' => 0]);
                $output->write($formatter->format($emptyReport));
            }

            return 0;
        }

        $patternCount = \count($patterns);
        $output->write('  '.$output->dim("Scanned {$fileCount} files, found {$patternCount} patterns.\n\n"));

        $progressCallback = null;
        $analysisStartTime = (float) microtime(true);
        if ('console' === $format && $config->shouldShowProgress()) {
            $output->write('  '.$output->dim('[2/2] Analyzing patterns')."\n");
            $output->progressStart(\count($patterns));
            $progressCallback = static fn () => $output->progressAdvance();
        }

        $report = $lint->analyze($patterns, $request, $progressCallback);

        if (null !== $progressCallback) {
            $output->progressFinish();
        }

        $analysisTime = (float) microtime(true) - $analysisStartTime;

        if ($arguments->baseline) {
            $baseline = $this->loadBaseline($arguments->baseline);
            $report = $this->filterReportByBaseline($report, $baseline);
        }

        $output->write($formatter->format($report));

        if ($formatter instanceof ConsoleFormatter) {
            $output->write($formatter->getSummary($report->stats));
            $elapsed = (float) microtime(true) - $startTime;
            $peakMemory = memory_get_peak_usage(true);
            $cacheStats = $regex->getCacheStats();
            $timeStr = round($elapsed, 2).'s';
            $memoryStr = round($peakMemory / 1024 / 1024, 2).' MB';
            $cacheStr = $cacheStats['hits'].' hits, '.$cacheStats['misses'].' misses';
            $processesStr = (string) $jobs;
            $output->write('  '.$output->dim('Time: ').$output->warning($timeStr).'  '.$output->dim('Memory: ').$output->warning($memoryStr).'  '.$output->dim('Cache: ').$output->warning($cacheStr).'  '.$output->dim('Processes: ').$output->warning($processesStr)."\n");
            $output->write($formatter->formatFooter());
        }

        if ($arguments->generateBaseline) {
            $baseline = $this->generateBaseline($report);
            file_put_contents($arguments->generateBaseline, json_encode($baseline, \JSON_PRETTY_PRINT | \JSON_UNESCAPED_SLASHES));
            $output->write("Baseline generated at {$arguments->generateBaseline}\n");
        }

        if ($arguments->baseline) {
            $baseline = $this->loadBaseline($arguments->baseline);
            $report = $this->filterReportByBaseline($report, $baseline);
        }

        if (null !== $outputFile) {
            $content = $formatter->format($report);
            if ('console' === $format) {
                // Strip ANSI codes for file output
                $content = preg_replace('/\e\[[0-9;]*m/', '', $content);
            }
            $dir = dirname($outputFile);
            if (!is_dir($dir) && !@mkdir($dir, 0o777, true) && !is_dir($dir)) {
                $output->write($output->error("Could not create directory: $dir\n"));
            } elseif (false === @file_put_contents($outputFile, $content)) {
                $output->write($output->error("Could not write to file: $outputFile\n"));
            } else {
                $output->write("Output also written to: {$outputFile}\n");
            }
        }

        return $report->stats['errors'] > 0 ? 1 : 0;
    }

    /**
     * @return array<array{file: string, line: int, message: string, type: string, pattern?: string|null}>
     */
    private function generateBaseline(RegexLintReport $report): array
    {
        $baseline = [];
        foreach ($report->results as $result) {
            foreach ($result['issues'] as $issue) {
                $relativeFile = $this->toRelativePath($issue['file']);
                $baseline[] = [
                    'file' => $relativeFile,
                    'line' => $issue['line'],
                    'message' => $issue['message'],
                    'type' => $issue['type'],
                    'pattern' => $issue['pattern'] ?? null,
                ];
            }
        }

        return $baseline;
    }

    /**
     * @return array<array{file: string, line: int, message: string, type: string, pattern?: string|null}>
     */
    private function loadBaseline(string $file): array
    {
        if (!file_exists($file)) {
            return [];
        }
        $content = file_get_contents($file);
        if (false === $content) {
            return [];
        }
        /** @var array<array{file: string, line: int, message: string, type: string, pattern?: string|null}> $data */
        $data = json_decode($content, true);

        return is_array($data) ? $data : [];
    }

    /**
     * @param array<array{file: string, line: int, message: string, type: string, pattern?: string|null}> $baseline
     */
    private function filterReportByBaseline(RegexLintReport $report, array $baseline): RegexLintReport
    {
        $baselineMap = [];
        foreach ($baseline as $item) {
            $key = $item['file'].':'.$item['line'].':'.$item['message'];
            $baselineMap[$key] = true;
        }

        $filteredResults = [];
        $errors = 0;
        $warnings = 0;
        $optimizations = 0;

        foreach ($report->results as $result) {
            $filteredIssues = [];
            foreach ($result['issues'] as $issue) {
                $relativeFile = $this->toRelativePath($issue['file']);
                $key = $relativeFile.':'.$issue['line'].':'.$issue['message'];
                if (!isset($baselineMap[$key])) {
                    $filteredIssues[] = $issue;
                    if ('error' === $issue['type']) {
                        $errors++;
                    } elseif ('warning' === $issue['type']) {
                        $warnings++;
                    } elseif ('optimization' === $issue['type']) {
                        $optimizations++;
                    }
                }
            }
            if (!empty($filteredIssues) || !empty($result['optimizations']) || !empty($result['problems'])) {
                $filteredResults[] = [
                    'file' => $result['file'],
                    'line' => $result['line'],
                    'source' => $result['source'] ?? null,
                    'pattern' => $result['pattern'] ?? null,
                    'location' => $result['location'] ?? null,
                    'issues' => $filteredIssues,
                    'optimizations' => $result['optimizations'],
                    'problems' => $result['problems'],
                ];
            }
        }

        return new RegexLintReport($filteredResults, [
            'errors' => $errors,
            'warnings' => $warnings,
            'optimizations' => $optimizations,
        ]);
    }

    private function toRelativePath(string $path): string
    {
        $normalizedPath = str_replace('\\', '/', $path);
        $cwd = getcwd();
        if (false === $cwd) {
            return $normalizedPath;
        }

        $normalizedCwd = rtrim(str_replace('\\', '/', $cwd), '/');
        if ('' === $normalizedCwd) {
            return $normalizedPath;
        }

        $prefix = $normalizedCwd.'/';
        if (str_starts_with($normalizedPath, $prefix)) {
            return substr($normalizedPath, \strlen($prefix));
        }

        return $normalizedPath;
    }

    /**
     * Detect the number of available CPU cores for optimal parallel processing.
     */
    private static function detectCpuCount(): int
    {
        // Try Swoole extension first (fastest)
        if (\function_exists('swoole_cpu_num')) {
            return swoole_cpu_num();
        }

        // Unix-like systems
        if (\DIRECTORY_SEPARATOR === '/') {
            // Linux
            if (\is_readable('/proc/cpuinfo')) {
                $cpuinfo = \file_get_contents('/proc/cpuinfo');
                if (false !== $cpuinfo) {
                    $matches = [];
                    \preg_match_all('/^processor\s*:/m', $cpuinfo, $matches);
                    if (!empty($matches[0])) {
                        return \count($matches[0]);
                    }
                }
            }

            // macOS/BSD
            $result = \shell_exec('sysctl -n hw.ncpu 2>/dev/null');
            if (null !== $result) {
                $cpu = (int) \trim((string) $result);
                if ($cpu > 0) {
                    return $cpu;
                }
            }
        } else {
            // Windows
            $result = \shell_exec('wmic cpu get NumberOfCores 2>nul | findstr /r /v "^$" | findstr /v "NumberOfCores"');
            if (null !== $result) {
                $cpu = (int) \trim((string) $result);
                if ($cpu > 0) {
                    return $cpu;
                }
            }
        }

        // Fallback
        return 1;
    }
}
